//Written for www.HomoFaciens.de
//Program simulates the behaviour of electrons and holes inside of a bipolar junction transistor.
//Copyright (C) 2011 Norbert Heinz
//
//This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation version 3 of the License.
//
//This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License along with this program; if not, see http://www.gnu.org/licenses/.

import java.awt.*;
import java.awt.event.*;
import java.awt.font.FontRenderContext;
import java.awt.geom.Rectangle2D;
import java.awt.image.*;
import java.net.URL;
import java.io.*;
import javax.swing.*;
import java.text.*;
import java.applet.*;
import java.util.Random;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.imageio.ImageIO;
import java.io.File;
import java.io.IOException;


public class BipolarJunctionTransistor extends JApplet implements  ActionListener, ChangeListener {

    static String versionNumber="v1.0";

    TransistorPainter painter;
    JSpinner crystalSizeSpinner, recombinationTimeSpinner, collectorVoltageSpinner, baseVoltageSpinner, electronMobilitySpinner, holeMobilitySpinner, temperatureSpinner;
    JButton startButton, restartButton, moveUpButton, moveDownButton, moveRightButton, moveLeftButton;
    JLabel crystalSizeLabel, recombinationTimeLabel, collectorVoltageLabel, baseVoltageLabel, electronMobilityLabel, holeMobilityLabel, temperatureLabel;
    JLabel crystalSizeUnitLabel, baseVoltageUnitLabel, electronMobilityUnitLabel, temperatureUnitLabel, collectorVoltageUnitLabel;
    JLabel sourceLabel;
    JCheckBox showStatsCheckBox;
    JRadioButton NPNButton, PNPButton;
    Timer timer;             //timer for simulation
    int timerPause=100;       //delay between two calculated frames in milliseconds
    static String language="initalValue";
    String givenParam="nothing yet";   //Arguments passed to applet
    static String ApplicationTitle="Bipolar junction transistor";
    Random RandomGenerator = new Random(1);
    int impurityPause=85;

    public void init() {
        UIManager.put("swing.boldMetal", Boolean.FALSE);

        try{
          givenParam = getParameter("LanguageSettings");
        }
        catch(NullPointerException npe){
          System.out.println("could not get parameter 'LanguageSettings'");
        }

        if(givenParam.compareTo("nothing yet")!=0) {
          if (givenParam.compareTo("German")==0)language="German";
          else language="English";
        }
    }

    public void start() {
      int i;
        initComponents();

        timer = new Timer(timerPause, new timerEvent());
    }

    public static void main(String[] args) {

        if(args.length>0){
          if (args[0].compareTo("German")==0)language="German";
          else language="English";
        }
        else{
          language="English";
        }

        if(language.compareTo("German")==0){
          ApplicationTitle="Bipolarer Transistor";
        }
        else{
          ApplicationTitle="Bipolar junction transistor";
        }

        JFrame f = new JFrame(ApplicationTitle);
        f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        JApplet ap = new BipolarJunctionTransistor();
        ap.init();
        ap.start();
        f.getContentPane().add("Center", ap);
        f.pack();
        f.setVisible(true);

    }

    private BufferedImage loadImage(String name) {
        String imgFileName = "pictures/"+name;
        URL url = BipolarJunctionTransistor.class.getResource(imgFileName);
        BufferedImage img = null;
        try {
            img =  ImageIO.read(url);
        }
        catch (Exception e){
          img=null;
          System.out.println("Error loading Image '" + imgFileName + "': " + e); // Display the string.
        }
        return img;
    }

    public void initComponents() {
        int i;
        JFormattedTextField ftf = null;
        GridBagConstraints c = new GridBagConstraints();
        Font font = new Font("Arial", Font.PLAIN, 12);

        getContentPane().setLayout(new BorderLayout());

        //create panels
        JPanel controlPanel = new JPanel();
        JPanel movePanel = new JPanel();
        JPanel scenePanel = new JPanel();
        JPanel mainPane = new JPanel();

        //set panel properties
        controlPanel.setLayout(new GridBagLayout());
        movePanel.setLayout(new GridBagLayout());
        mainPane.setLayout(new BoxLayout(mainPane, BoxLayout.PAGE_AXIS));
        mainPane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

        //create buttons, labels etcetera
        if(language.compareTo("German")==0){
          startButton = new JButton("Start");

          restartButton = new JButton("Neustart");

          moveUpButton = new JButton("Hoch");
          moveDownButton = new JButton("Runter");
          moveLeftButton = new JButton("Links");
          moveRightButton = new JButton("Rechts");

          collectorVoltageLabel = new JLabel("Kollektorspannung:");
          crystalSizeLabel = new JLabel("Größe:");
          recombinationTimeLabel = new JLabel("Rekombinationszeit:");
          baseVoltageLabel = new JLabel("Basisspannung");
          electronMobilityLabel = new JLabel("Elekt. Mob.");
          holeMobilityLabel = new JLabel("Lochbewegl.");
          temperatureLabel = new JLabel("Temperatur");
          showStatsCheckBox = new JCheckBox("Statistik anzeigen");
        }
        else if(language.compareTo("XXX")==0){
          crystalSizeLabel = new JLabel("XXX:");
        }
        else{
          startButton = new JButton("Start");

          restartButton = new JButton("Restart");

          moveUpButton = new JButton("Up");
          moveDownButton = new JButton("Down");
          moveLeftButton = new JButton("Left");
          moveRightButton = new JButton("Right");

          collectorVoltageLabel = new JLabel("Collector voltage:");
          crystalSizeLabel = new JLabel("Size:");
          recombinationTimeLabel = new JLabel("Recombination Time:");
          baseVoltageLabel = new JLabel("Base voltage");
          electronMobilityLabel = new JLabel("Elect. mob.");
          holeMobilityLabel = new JLabel("Hole mobility");
          temperatureLabel = new JLabel("Temperature");
          showStatsCheckBox = new JCheckBox("Show statistic");
        }

        NPNButton = new JRadioButton("NPN");
        PNPButton = new JRadioButton("PNP");

        crystalSizeUnitLabel = new JLabel("%");
        baseVoltageUnitLabel = new JLabel("%");
        electronMobilityUnitLabel = new JLabel("%");
        temperatureUnitLabel = new JLabel("%");
        collectorVoltageUnitLabel = new JLabel("%");
        sourceLabel = new JLabel(versionNumber + "   Source & info:    www.HomoFaciens.de");

        showStatsCheckBox.setFont(font);
        showStatsCheckBox.setSelected(true);
        showStatsCheckBox.setActionCommand("showStatsPressed");
        showStatsCheckBox.addActionListener(this);

        NPNButton.setFont(font);
        NPNButton.setActionCommand("NPNButtonPressed");
        NPNButton.setSelected(true);
        NPNButton.addActionListener(this);

        PNPButton.setFont(font);
        PNPButton.setActionCommand("PNPButtonPressed");
        PNPButton.setSelected(false);
        PNPButton.addActionListener(this);

        ButtonGroup group = new ButtonGroup();
        group.add(NPNButton);
        group.add(PNPButton);


        SpinnerNumberModel baseVoltageModel = new SpinnerNumberModel(0, 0, 100, 10);
        baseVoltageSpinner = new JSpinner(baseVoltageModel);
        ftf = getTextField(baseVoltageSpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        baseVoltageSpinner.addChangeListener(this);

        SpinnerNumberModel collectorVoltageModel = new SpinnerNumberModel(0, 0, 100, 10);
        collectorVoltageSpinner = new JSpinner(collectorVoltageModel);
        ftf = getTextField(collectorVoltageSpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        collectorVoltageSpinner.addChangeListener(this);

        SpinnerNumberModel electronMobilityModel = new SpinnerNumberModel(18, 1, 20, 1);
        electronMobilitySpinner = new JSpinner(electronMobilityModel);
        ftf = getTextField(electronMobilitySpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        electronMobilitySpinner.addChangeListener(this);

        SpinnerNumberModel holeMobilityModel = new SpinnerNumberModel(10, 1, 20, 1);
        holeMobilitySpinner = new JSpinner(holeMobilityModel);
        ftf = getTextField(holeMobilitySpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        holeMobilitySpinner.addChangeListener(this);

        SpinnerNumberModel crystalSizeModel = new SpinnerNumberModel(5, 5, 100, 1);
        crystalSizeSpinner = new JSpinner(crystalSizeModel);
        ftf = getTextField(crystalSizeSpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        crystalSizeSpinner.addChangeListener(this);
        crystalSizeLabel.setLabelFor(crystalSizeSpinner);

        SpinnerNumberModel recombinationTimeModel = new SpinnerNumberModel(15, 0, 100, 1);
        recombinationTimeSpinner = new JSpinner(recombinationTimeModel);
        ftf = getTextField(recombinationTimeSpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        recombinationTimeSpinner.addChangeListener(this);

        SpinnerNumberModel temperatureModel = new SpinnerNumberModel(10, 1, 100, 1);
        temperatureSpinner = new JSpinner(temperatureModel);
        ftf = getTextField(temperatureSpinner);
        if (ftf != null ) {
            ftf.setFont(font);
            ftf.setColumns(4); //specify more width than we need
            ftf.setHorizontalAlignment(JTextField.RIGHT);
        }
        temperatureSpinner.addChangeListener(this);


        painter = new TransistorPainter();

        painter.atomSilicon = loadImage("silicon.png");
        painter.atomLightBlueUp = loadImage("silicon-lightblue-up.png");
        painter.atomLightBlueDown = loadImage("silicon-lightblue-down.png");
        painter.atomLightRedUp = loadImage("silicon-lightred-up.png");
        painter.atomLightRedDown = loadImage("silicon-lightred-down.png");
        painter.atomDarkBlueUp = loadImage("silicon-darkblue-up.png");
        painter.atomDarkBlueDown = loadImage("silicon-darkblue-down.png");
        painter.atomDarkRedUp = loadImage("silicon-darkred-up.png");
        painter.atomDarkRedDown = loadImage("silicon-darkred-down.png");
        painter.atomGreyUp = loadImage("aluminum-up.png");
        painter.atomGreyDown = loadImage("aluminum-down.png");
        painter.atomGoldUp = loadImage("phosphorus-up.png");
        painter.atomGoldDown = loadImage("phosphorus-down.png");
        painter.recombination = loadImage("recombination.png");


        //set properties of controls
        crystalSizeUnitLabel.setFont(font);
        collectorVoltageUnitLabel.setFont(font);

        collectorVoltageLabel.setFont(font);
        crystalSizeLabel.setFont(font);
        recombinationTimeLabel.setFont(font);

        sourceLabel.setFont(font);

        startButton.setFont(font);
        startButton.setActionCommand("startButtonPressed");
        startButton.addActionListener(this);

        restartButton.setFont(font);
        restartButton.setActionCommand("restartButtonPressed");
        restartButton.addActionListener(this);

        moveUpButton.setFont(font);
        moveUpButton.setActionCommand("moveUpButtonPressed");
        moveUpButton.addActionListener(this);

        moveDownButton.setFont(font);
        moveDownButton.setActionCommand("moveDownButtonPressed");
        moveDownButton.addActionListener(this);

        moveLeftButton.setFont(font);
        moveLeftButton.setActionCommand("moveLeftButtonPressed");
        moveLeftButton.addActionListener(this);

        moveRightButton.setFont(font);
        moveRightButton.setActionCommand("moveRightButtonPressed");
        moveRightButton.addActionListener(this);


        //add components

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;
        c.gridx = 1;
        c.gridy = 0;
        c.gridwidth=1;
        movePanel.add(moveUpButton, c);

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;
        c.gridx = 0;
        c.gridy = 1;
        c.gridwidth=1;
        movePanel.add(moveLeftButton, c);

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;
        c.gridx = 2;
        c.gridy = 1;
        c.gridwidth=1;
        movePanel.add(moveRightButton, c);

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;
        c.gridx = 1;
        c.gridy = 2;
        c.gridwidth=1;
        movePanel.add(moveDownButton, c);

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;
        c.gridx = 9;
        c.gridy = 1;
        c.gridwidth=3;
        c.gridheight=3;
        controlPanel.add(movePanel, c);


        c.fill = GridBagConstraints.HORIZONTAL;
        c.ipadx = 10;
        c.insets = new Insets(5,5,5,5);  //padding
        c.gridheight=1;


        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor=GridBagConstraints.CENTER;

        c.gridx = 9;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(crystalSizeLabel, c);

        c.gridx = 10;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(crystalSizeSpinner, c);

        c.gridx = 11;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(crystalSizeUnitLabel, c);

        c.gridx = 3;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(temperatureLabel, c);

        c.gridx = 4;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(temperatureSpinner, c);

        c.gridx = 5;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(temperatureUnitLabel, c);

        c.gridx = 6;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(recombinationTimeLabel, c);

        c.gridx = 7;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(recombinationTimeSpinner, c);

        c.gridx = 0;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(collectorVoltageLabel, c);

        c.gridx = 1;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(collectorVoltageSpinner, c);

        c.gridx = 2;
        c.gridy = 0;
        c.gridwidth=1;
        controlPanel.add(collectorVoltageUnitLabel, c);

        c.gridx = 0;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(baseVoltageLabel, c);

        c.gridx = 1;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(baseVoltageSpinner, c);

        c.gridx = 2;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(baseVoltageUnitLabel, c);

        c.gridx = 3;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(electronMobilityLabel, c);

        c.gridx = 4;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(electronMobilitySpinner, c);

        c.gridx = 5;
        c.gridy = 1;
        c.gridwidth=1;
        //controlPanel.add(electronMobilityUnitLabel, c);

        c.gridx = 6;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(holeMobilityLabel, c);

        c.gridx = 7;
        c.gridy = 1;
        c.gridwidth=1;
        controlPanel.add(holeMobilitySpinner, c);

        c.gridx = 8;
        c.gridy = 1;
        c.gridwidth=1;
        //controlPanel.add(electronMobilityUnitLabel, c);

        c.gridx = 0;
        c.gridy = 2;
        c.gridwidth=1;
        controlPanel.add(showStatsCheckBox, c);

        c.gridx = 6;
        c.gridy = 2;
        c.gridwidth=1;
        controlPanel.add(NPNButton, c);

        c.gridx = 7;
        c.gridy = 2;
        c.gridwidth=1;
        controlPanel.add(PNPButton, c);


        c.gridx = 1;
        c.gridy = 2;
        c.gridwidth=2;
        controlPanel.add(startButton, c);
        c.gridx = 3;
        c.gridy = 2;
        c.gridwidth=2;
        controlPanel.add(restartButton, c);


        c.gridx = 6;
        c.gridy = 3;
        c.gridwidth=3;
        c.fill = GridBagConstraints.NONE;
        c.anchor=GridBagConstraints.LINE_END;
        controlPanel.add(sourceLabel, c);


        //setup panels
        scenePanel.add("Center", painter);

        mainPane.add(controlPanel);
        mainPane.add(scenePanel);
        getContentPane().add(mainPane);
        getRootPane().setDefaultButton(startButton);


        //initialize crystal

        initCrystal();

    }

    public JFormattedTextField getTextField(JSpinner spinner) {
        JComponent editor = spinner.getEditor();
        if (editor instanceof JSpinner.DefaultEditor) {
            return ((JSpinner.DefaultEditor)editor).getTextField();
        } else {
            System.err.println("Unexpected editor type: "  + spinner.getEditor().getClass() + " isn't a descendant of DefaultEditor");
            return null;
        }
    }

    public void initValues(){
      painter.crystalSize = ((Integer)crystalSizeSpinner.getValue()).doubleValue() * painter.sceneWidth / (1920 / 2) * 2.2;
      painter.recombinationTime = ((Integer)recombinationTimeSpinner.getValue()).intValue();
      painter.collectorVoltage = (int)((Integer)collectorVoltageSpinner.getValue()).doubleValue();
      painter.collectorVoltage = (int)(painter.collectorVoltage / 10) * 10;
      collectorVoltageSpinner.setValue(painter.collectorVoltage);
      painter.holeMobility = 21 - (int)((Integer)holeMobilitySpinner.getValue()).doubleValue();
      painter.baseVoltage = (int)((Integer)baseVoltageSpinner.getValue()).intValue();
      painter.baseVoltage = (int)(painter.baseVoltage / 10) * 10;
      baseVoltageSpinner.setValue(painter.baseVoltage);
      painter.electronMobility = 21 - (int)((Integer)electronMobilitySpinner.getValue()).intValue();
      painter.crystalTemperature = (int)((Integer)temperatureSpinner.getValue()).doubleValue();
      painter.showStats=showStatsCheckBox.isSelected();
    }

    public void initCrystal(){
      double RowX, RowY, Spacer;
      boolean halfSpace=false;
      int i, i2, i3, i4, i5, j;

      initValues();

      Spacer=(50.0);

        if(painter.transistorType=="PNP"){
          RandomGenerator.setSeed(2);
        }

        i2=0;
        i3=0;
        i4=2;
        i5=0;
        painter.electronNumber=0;
        painter.holeNumber=0;
        RowX = 0;
        RowY = 3 * Math.cos(60.0 * 3.1415927 / 180.0) * Spacer + 2 * Spacer;
        for(i=0;i<10;i++){
          painter.impuritySumZones[i]=0;
        }
        for(i=0;i<12200;i++){
          j=i;
          while(j>200)j-=200;
          if(j>99){
            painter.AtomIsUp[i] = 1;
          }
          else{
            painter.AtomIsUp[i]= 0;
          }

          j=i;
          while(j>99)j-=100;

          painter.AtomCoords[i][0]=RowX;
          painter.AtomCoords[i][1]=RowY;
          RowX+=Math.sin(60.0 * 3.1415927 / 180.0) * Spacer * 2;
          i3++;
          if(i3==100){
            i3=0;
            if(i2==0 | i2==1){
              RowX=Math.sin(60.0 * 3.1415927 / 180.0) * Spacer;
            }
            else{
              RowX=0;
            }
            if(i2==0 | i2==2){
              RowY += Math.cos(60.0 * 3.1415927 / 180.0) * Spacer;
            }
            else{
              RowY += Spacer;
            }
            i2++;
            if(i2==4)i2=0;
          }
          painter.directionState[i]=i2;
          painter.AtomType[i]=0;     //Silicon atoms
          painter.AtomElectron[i]=0; //Normal number of electrons
          painter.AtomMoveNow[i]=0;  //Time electron/hole needs to move
          i4++;
          if(i4==impurityPause){
            i4=RandomGenerator.nextInt(3)-1; //randomize distribution of impurity atoms slightly
//            i4=0;
            painter.chargePlaces[i5] = i;
            painter.chargePlacesNew[i5][0] = i;
          if(((j < 60 - painter.middleLayer | j > 60) && painter.transistorType=="NPN") |
              (!(j < 60 - painter.middleLayer | j > 60) && painter.transistorType=="PNP") ){
              painter.chargePlacesNew[i5][1] = 1;
              painter.AtomMoveNow[i] = RandomGenerator.nextInt(2);
            }
            else{
              painter.chargePlacesNew[i5][1] = 2;
              painter.AtomMoveNow[i] = RandomGenerator.nextInt(6);
            }
            painter.chargePlacesNew[i5][2] = 0;//Recombination not blocked
            painter.impurityPlaces[i5] = i;
            i5++;
          if(((j < 60 - painter.middleLayer | j > 60) && painter.transistorType=="NPN") |
              (!(j < 60 - painter.middleLayer | j > 60) && painter.transistorType=="PNP") ){
              painter.AtomType[i]=1;      //Phosphorous atoms
              painter.AtomElectron[i]=1;  //Additional electron
              painter.electronNumber++;
              if(j < 10){
                painter.impuritySumZones[0]++;
              }
              if(j >= 10 && j < 20){
                painter.impuritySumZones[1]++;
              }
              if(j >= 20 && j < 30){
                painter.impuritySumZones[2]++;
              }
              if(j >= 30 && j < 40){
                painter.impuritySumZones[3]++;
              }
              if(j >= 40 && j < 50){
                painter.impuritySumZones[4]++;
              }
              if(j >= 50 && j < 60){
                painter.impuritySumZones[5]++;
              }
              if(j >= 60 && j < 70){
                painter.impuritySumZones[6]++;
              }
              if(j >= 70 && j < 80){
                painter.impuritySumZones[7]++;
              }
              if(j >= 80 && j < 90){
                painter.impuritySumZones[8]++;
              }
              if(j >= 90){
                painter.impuritySumZones[9]++;
              }
            }
            else{
              painter.AtomType[i]=2;     //Aluminum atoms
              painter.AtomElectron[i]=2; //Missing electron
              painter.holeNumber++;
              if(j < 10){
                painter.impuritySumZones[0]--;
              }
              if(j >= 10 && j < 20){
                painter.impuritySumZones[1]--;
              }
              if(j >= 20 && j < 30){
                painter.impuritySumZones[2]--;
              }
              if(j >= 30 && j < 40){
                painter.impuritySumZones[3]--;
              }
              if(j >= 40 && j < 50){
                painter.impuritySumZones[4]--;
              }
              if(j >= 50 && j < 60){
                painter.impuritySumZones[5]--;
              }
              if(j >= 60 && j < 70){
                painter.impuritySumZones[6]--;
              }
              if(j >= 70 && j < 80){
                painter.impuritySumZones[7]--;
              }
              if(j >= 80 && j < 90){
                painter.impuritySumZones[8]--;
              }
              if(j >= 90){
                painter.impuritySumZones[9]--;
              }
            }
          }
        }//for(i=0;i<12200;i++){
        for(i=0;i<10;i++){
          painter.chargeSumZones[i] = -painter.impuritySumZones[i];
        }
        painter.chargeNumber=i5;
        painter.impurityNumber=i5;

        for(i=0;i<100;i++){
          painter.electronInjectTime[i]=0;
        }


   }

    public void stateChanged(ChangeEvent e) {
      initValues();
      painter.refreshScene();
    }

    // Listens to the buttons.
    public void actionPerformed(ActionEvent e) {

        if("showStatsPressed".equals(e.getActionCommand())) {
          if(!painter.isRunning){
            painter.showStats=showStatsCheckBox.isSelected();
            painter.refreshScene();
          }
        }
        if("moveDownButtonPressed".equals(e.getActionCommand())) {
          painter.upperLeftCornerY+=25;
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("moveUpButtonPressed".equals(e.getActionCommand())) {
          painter.upperLeftCornerY-=25;
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("moveLeftButtonPressed".equals(e.getActionCommand())) {
          painter.upperLeftCornerX-=25;
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("moveRightButtonPressed".equals(e.getActionCommand())) {
          painter.upperLeftCornerX+=25;
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("NPNButtonPressed".equals(e.getActionCommand())) {
          painter.transistorType = "NPN";
          painter.middleLayer = 20;
          initCrystal();
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("PNPButtonPressed".equals(e.getActionCommand())) {
          painter.transistorType = "PNP";
          painter.middleLayer = 12;
          initCrystal();
          if(!painter.isRunning){
            painter.refreshScene();
          }
        }
        if("restartButtonPressed".equals(e.getActionCommand())) {
          int i, i2;
          Random RandomGenerator = new Random();
          painter.electronNumber=0;
          painter.holeNumber=0;
          for(i=0;i<12200;i++){
            painter.AtomElectron[i]=0; //Normal number of electrons
            painter.AtomMoveNow[i]=0; //Time electron/hole needs to move
          }
          for(i=0;i<painter.impurityNumber;i++){
            painter.chargePlacesNew[i][0] = painter.impurityPlaces[i];
            painter.chargePlaces[i] =  painter.impurityPlaces[i];
            if (painter.AtomType[painter.impurityPlaces[i]]==1){
              painter.AtomElectron[painter.impurityPlaces[i]]=1;
              painter.electronNumber++;
              painter.chargePlacesNew[i][1] = 1;
              painter.AtomMoveNow[painter.impurityPlaces[i]] = RandomGenerator.nextInt(2);
            }
            if (painter.AtomType[painter.impurityPlaces[i]]==2){
              painter.AtomElectron[painter.impurityPlaces[i]]=2;
              painter.holeNumber++;
              painter.chargePlacesNew[i][1] = 2;
              painter.AtomMoveNow[painter.impurityPlaces[i]] = RandomGenerator.nextInt(6);
            }
          }
          for(i=0;i<100;i++){
            painter.electronInjectTime[i]=0;
          }
          painter.chargeNumber=painter.impurityNumber;
          painter.refreshScene();
        }
        if("startButtonPressed".equals(e.getActionCommand())) {
          Random RandomGenerator = new Random();
          if(painter.isRunning){
            painter.isRunning=false;
          }
          else{
            if(language.compareTo("German")==0){
              startButton.setText("Stopp");
            }
            else if(language.compareTo("XXX")==0){
              startButton.setText("Stop");
            }
            else{
              startButton.setText("Stop");
            }
            if(timer.getInitialDelay()!=timerPause){
              timer.setInitialDelay(timerPause);
            }
            timer.start();
            painter.isRunning=true;
          }//if(!isRunning)
        }//if ("startButtonPressed".equals(e.getActionCommand())) {
    }

    private class timerEvent implements ActionListener{
      public void actionPerformed(ActionEvent e){

        if(!painter.isRunning){
          timer.stop();
          if(language.compareTo("German")==0){
            startButton.setText("Start");
          }
          else if(language.compareTo("XXX")==0){
            startButton.setText("Start");
          }
          else{
            startButton.setText("Start");
          }
        }

        initValues();

        painter.refreshScene();
      }
    }
}



class TransistorPainter extends Component {

    int sceneHeight = 1080 / 2, sceneWidth = 1920 / 2;
    double[][] AtomCoords = new double[12200][2];
    int[] AtomIsUp = new int[12200];
    int[] AtomElectron = new int[12200];
    int[] AtomType = new int[12200];
    int[] AtomMoveNow = new int[12200];
    int chargeNumber = 0, impurityNumber = 0;
    int[] directionState = new int[12200];
    int[] chargePlaces = new int[12200];
    int[] impurityPlaces = new int[12200];
    int[][] chargePlacesNew = new int[12200][3];
    double[] electronInjectX = new double[100];
    double[] electronInjectY = new double[100];
    int[] electronInjectTime = new int[100];
    int[] chargeSumZones = new int[10];
    int[] impuritySumZones = new int[10];
    double crystalSize = 20.0, crystalSize2 = 21.0;
    int recombinationTime = 5;
    double crystalTemperature = 0.0;
    int collectorVoltage = 0,  collectorVoltageOld = 0;
    int holeMobility = 15;
    int baseVoltage = 0;
    int middleLayer = 20;
    int electronMobility = 10;
    int electronNumber = 0, holeNumber = 0;
    BufferedImage bufferImage = null, bufferImage2 = null;
    Graphics2D bufferImageSurface = null, bufferImageSurface2 = null;
    boolean isRunning=false;
    boolean showStats=true;
    BufferedImage atomSilicon = null, atomLightRedUp = null, atomLightRedDown = null, atomLightBlueUp = null, atomLightBlueDown = null, atomDarkBlueUp = null, atomDarkBlueDown = null;
    BufferedImage atomDarkRedUp = null, atomDarkRedDown = null, atomGreyUp = null, atomGreyDown = null, atomGoldUp = null, atomGoldDown = null, recombination = null;
    String picName = "drawing.jpg";
    String transistorType="NPN", transistorTypeOld="NPN";
    int picNumber = 0;
    int savePics = 0;
    int vibration = 0;
    double vibrateX = 0.0, vibrateY = 0.0;
    int upperLeftCornerX = 30, upperLeftCornerY = 0, upperLeftCornerXold = 0, upperLeftCornerYold = 0, baseVoltageOld = 0;

    public Dimension getPreferredSize(){
        return new Dimension(sceneWidth, sceneHeight);
    }

    void refreshScene() {
      repaint();
    }

    public void paint(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        Dimension size = getSize();
        Composite origComposite;
        int i, i2, i3 = 0, i4, i5, iNew, j;
        double TopLeftX = 0.0, TopLeftY = 0.0;
        double forceX = 0.0, forceY = 0.0, totalDistance = 0.0, totalForce = 0.0;
        double alpha = 0.0;
        int newElectron=0, newHole=0, removeElectron=0;
        int tempInt=0, xCoord;
        int[] triangleX = new int[3];
        int[] triangleY = new int[3];

        //create buffer to avoid flickering while scene is calculated and painted
        if(bufferImage==null){
          bufferImage = (BufferedImage)createImage(sceneWidth, sceneHeight);
          bufferImageSurface = bufferImage.createGraphics();
          bufferImage2 = (BufferedImage)createImage(sceneWidth, sceneHeight);
          bufferImageSurface2 = bufferImage2.createGraphics();
          Font font = new Font("Arial", Font.PLAIN, 26);
          bufferImageSurface.setFont(font);
        }

        origComposite = g2.getComposite();

        Random RandomGenerator = new Random();


        //Repaint silicon crystal whenever view was changed
        if(crystalSize != crystalSize2 | upperLeftCornerXold != upperLeftCornerX | upperLeftCornerYold != upperLeftCornerY | collectorVoltage != collectorVoltageOld | baseVoltage != baseVoltageOld | vibration == 1 | transistorType != transistorTypeOld){
          bufferImageSurface2.setBackground(new Color(217, 217, 217));
          bufferImageSurface2.clearRect(0, 0, sceneWidth, sceneHeight);
          if (baseVoltage==0){//Paint base terminals
            bufferImageSurface2.setColor(new Color(200, 200, 200));
          }
          else{
            if(transistorType=="PNP"){
              bufferImageSurface2.setColor(new Color(150, 150, 150));
            }
            else{
              bufferImageSurface2.setColor(new Color(255, 150, 150));
            }
          }
          if(transistorType=="PNP"){
            bufferImageSurface2.fillRect( (int)((4400 + upperLeftCornerX) * crystalSize / 100.0),
                                          (int)(upperLeftCornerY * crystalSize / 100.0),
                                          (int)(750 * crystalSize / 100.0),
                                          (int)(230 * crystalSize / 100.0));
            bufferImageSurface2.fillRect( (int)((4400 + upperLeftCornerX) * crystalSize / 100.0),
                                          (int)((4750 + upperLeftCornerY) * crystalSize / 100.0),
                                          (int)(750 * crystalSize / 100.0),
                                          (int)(230 * crystalSize / 100.0));
          }
          else{
            bufferImageSurface2.fillRect( (int)((4000 + upperLeftCornerX) * crystalSize / 100.0),
                                          (int)(upperLeftCornerY * crystalSize / 100.0),
                                          (int)(1000 * crystalSize / 100.0),
                                          (int)(230 * crystalSize / 100.0));
            bufferImageSurface2.fillRect( (int)((4000 + upperLeftCornerX) * crystalSize / 100.0),
                                          (int)((4750 + upperLeftCornerY) * crystalSize / 100.0),
                                          (int)(1000 * crystalSize / 100.0),
                                          (int)(230 * crystalSize / 100.0));
          }
          if (collectorVoltage == 0 && baseVoltage == 0){//Paint source and collector terminals
            bufferImageSurface2.setColor(new Color(200, 200, 200));
          }
          else{
            if(transistorType=="NPN"){
              bufferImageSurface2.setColor(new Color(150, 150, 150));
            }
            else{
              bufferImageSurface2.setColor(new Color(255, 150, 150));
            }
          }//if (collectorVoltage==0){
          bufferImageSurface2.fillRect( (int)(upperLeftCornerX * crystalSize / 100.0),
                                        (int)(upperLeftCornerY * crystalSize / 100.0),
                                        (int)(200 * crystalSize / 100.0),
                                        (int)(5000 * crystalSize / 100.0));
          if (collectorVoltage==0){//Paint source and collector terminals
            bufferImageSurface2.setColor(new Color(200, 200, 200));
          }
          else{
            if(transistorType=="PNP"){
              bufferImageSurface2.setColor(new Color(150, 150, 150));
            }
            else{
              bufferImageSurface2.setColor(new Color(255, 150, 150));
            }
          }//if (collectorVoltage==0){
          bufferImageSurface2.fillRect( (int)((upperLeftCornerX + 8470) * crystalSize / 100.0),
                                        (int)(upperLeftCornerY * crystalSize / 100.0),
                                        (int)(200 * crystalSize / 100.0),
                                        (int)(5000 * crystalSize / 100.0));
          for(i=0;i<12200;i++){
              if(vibration==1){
                vibrateX = (RandomGenerator.nextInt(100) - 50) / 20.0;
                vibrateY = (RandomGenerator.nextInt(100) - 50) / 20.0;
              }
              bufferImageSurface2.drawImage(atomSilicon,
                                          (int)((AtomCoords[i][0] + upperLeftCornerX + vibrateX) * crystalSize / 100.0),
                                          (int)((AtomCoords[i][1] + upperLeftCornerY + vibrateY) * crystalSize / 100.0),
                                          (int) (0.6 * crystalSize),
                                          (int) (0.6 * crystalSize),
                                          this);
              j=i;
              while(j>200)j-=200;
              if(j==99){
                i+=100;
              }
          }
          for(i=100;i<12200;i++){
              bufferImageSurface2.drawImage(atomSilicon,
                                          (int)((AtomCoords[i][0] + upperLeftCornerX + vibrateX) * crystalSize / 100.0),
                                          (int)((AtomCoords[i][1] + upperLeftCornerY + vibrateY) * crystalSize / 100.0),
                                          (int) (0.6 * crystalSize),
                                          (int) (0.6 * crystalSize),
                                          this);
              j=i;
              while(j>200)j-=200;
              if(j==199){
                i+=100;
              }
          }
          upperLeftCornerXold = upperLeftCornerX;
          upperLeftCornerYold = upperLeftCornerY;
          baseVoltageOld = baseVoltage;
          collectorVoltageOld = collectorVoltage;
          transistorTypeOld = transistorType;
        }

        bufferImageSurface.drawImage(bufferImage2, 0, 0, sceneWidth, sceneHeight, this);

        //painting impurity atoms, extra electrons and holes
        if(vibration == 0){
          for(i=0;i<12200;i++){
            if((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0 >= TopLeftX &&
              (AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0 <= sceneWidth &&
              (AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0 >= TopLeftY &&
              (AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0 <= sceneHeight){

              if(AtomType[i]==1.0){
                if(AtomIsUp[i]==1){
                  bufferImageSurface.drawImage(atomDarkRedUp,
                                              (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                              (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                              (int) (0.6 * crystalSize),
                                              (int) (0.6 * crystalSize),
                                              this);
                }
                else{
                  bufferImageSurface.drawImage(atomDarkRedDown,
                                              (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                              (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                              (int) (0.6 * crystalSize),
                                              (int) (0.6 * crystalSize),
                                              this);
                }
              }
              if(AtomType[i]==2.0){
                if(AtomIsUp[i]==1){
                  bufferImageSurface.drawImage(atomDarkBlueUp,
                                              (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                              (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                              (int) (0.6 * crystalSize),
                                              (int) (0.6 * crystalSize),
                                              this);
                }
                else{
                  bufferImageSurface.drawImage(atomDarkBlueDown,
                                              (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                              (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                              (int) (0.6 * crystalSize),
                                              (int) (0.6 * crystalSize),
                                              this);
                }
              }
              if(AtomElectron[i]==2.0){
                if(AtomType[i]==2.0){//Positive aluminum atom
                  if(AtomIsUp[i]==1){
                    bufferImageSurface.drawImage(atomGreyUp,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                  else{
                    bufferImageSurface.drawImage(atomGreyDown,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                }
                else{
                  if(AtomIsUp[i]==1){
                    bufferImageSurface.drawImage(atomLightRedUp,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                  else{
                    bufferImageSurface.drawImage(atomLightRedDown,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                }
              }
              if(AtomElectron[i]==1.0){
                if(AtomType[i]==1.0){//Negative phosphorus atom
                  if(AtomIsUp[i]==1){
                    bufferImageSurface.drawImage(atomGoldUp,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                  else{
                    bufferImageSurface.drawImage(atomGoldDown,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                }
                else{
                  if(AtomIsUp[i]==1){
                    bufferImageSurface.drawImage(atomLightBlueUp,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                  else{
                    bufferImageSurface.drawImage(atomLightBlueDown,
                                                (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0),
                                                (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0),
                                                (int) (0.6 * crystalSize),
                                                (int) (0.6 * crystalSize),
                                                this);
                  }
                }
              }
              if(AtomMoveNow[i]<0){
                  bufferImageSurface.drawImage(recombination,
                                              (int)((AtomCoords[i][0] + upperLeftCornerX) * crystalSize / 100.0 - 0.6 * crystalSize * 0.75),
                                              (int)((AtomCoords[i][1] + upperLeftCornerY) * crystalSize / 100.0 - 0.6 * crystalSize * 0.75),
                                              (int) (0.6 * crystalSize * 3.0),
                                              (int) (0.6 * crystalSize * 3.0),
                                              this);
              }
            }//if atom is visible

            if(AtomMoveNow[i]>0){
              AtomMoveNow[i]--;
            }
            if(AtomMoveNow[i]<0 && recombinationTime < 100){
              AtomMoveNow[i]++;
            }

            if(i<100){//draw triangles to show flow of electrons
              if(electronInjectTime[i]>0){
                if(electronInjectX[i]>AtomCoords[10][0] && electronInjectX[i]<AtomCoords[90][0]){//Electrons removed at base junction
                  triangleX[0]=(int) ((electronInjectX[i] + upperLeftCornerX) * crystalSize / 100.0);
                  triangleX[1]=(int)(triangleX[0] - 70 * crystalSize / 100.0);
                  triangleX[2]=(int)(triangleX[0] + 70 * crystalSize / 100.0);
                  triangleY[0]=(int)((electronInjectY[i] + upperLeftCornerY) * crystalSize / 100.0);
                  triangleY[1]=(int)(triangleY[0] - 200 * crystalSize / 100.0);
                  triangleY[2]=triangleY[1];
                  if(electronInjectY[i]==200){
                    triangleY[1]=triangleY[0];
                    triangleY[0]=triangleY[2];
                    triangleY[2]=triangleY[1];
                  }
                  if (transistorType=="PNP"){
                    triangleY[1]=triangleY[0];
                    triangleY[0]=triangleY[2];
                    triangleY[2]=triangleY[1];
                  }
                }
                else{//electrons injected at source / removed at collector
                  triangleX[0]=(int) ((electronInjectX[i] + upperLeftCornerX) * crystalSize / 100.0);
                  triangleX[1]=(int)(triangleX[0] + 200 * crystalSize / 100.0);
                  triangleX[2]=triangleX[0];
                  triangleY[1]=(int)((electronInjectY[i] + upperLeftCornerY) * crystalSize / 100.0);
                  triangleY[0]=(int)(triangleY[1] + 100 * crystalSize / 100.0);
                  triangleY[2]=(int)(triangleY[1] - 100 * crystalSize / 100.0);
                  if (transistorType=="PNP"){
                    triangleX[0]=triangleX[1];
                    triangleX[1]=triangleX[2];
                    triangleX[2]=triangleX[0];
                  }
                }
                electronInjectTime[i]--;
                bufferImageSurface.setColor(new Color(100, 100, 217));
                bufferImageSurface.fillPolygon( triangleX, triangleY, 3);
              }
            }
          }//if(vibration==0)
        }//for(i=0;i<12200;i++){

      if(savePics == 1){
        picNumber++;
        picName="picsequence/drawing-nostat-" + new DecimalFormat("000000").format(picNumber) + ".png";
        try{
          ImageIO.write(bufferImage, "PNG", new File(picName));
        }
        catch (IOException e1) {
          e1.printStackTrace();
        }
      }


        if(showStats){
          bufferImageSurface.setColor(new Color(0, 150, 0));
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[0] + chargeSumZones[0]), (int)((57 + 865 * 0.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[1] + chargeSumZones[1]), (int)((57 + 865 * 1.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[2] + chargeSumZones[2]), (int)((57 + 865 * 2.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[3] + chargeSumZones[3]), (int)((57 + 865 * 3.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[4] + chargeSumZones[4]), (int)((57 + 865 * 4.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[5] + chargeSumZones[5]), (int)((57 + 865 * 5.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[6] + chargeSumZones[6]), (int)((57 + 865 * 6.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[7] + chargeSumZones[7]), (int)((57 + 865 * 7.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[8] + chargeSumZones[8]), (int)((57 + 865 * 8.0 + upperLeftCornerX) * crystalSize / 100.0), 50);
          bufferImageSurface.drawString("Q = " + new DecimalFormat("#").format(impuritySumZones[9] + chargeSumZones[9]), (int)((57 + 865 * 9.0 + upperLeftCornerX) * crystalSize / 100.0), 50);

          bufferImageSurface.fillRect((int)((8 + 865 * 1.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 2.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 3.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 4.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 5.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 6.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 7.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 8.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
          bufferImageSurface.fillRect((int)((8 + 865 * 9.0 + upperLeftCornerX) * crystalSize / 100.0), 0, 3, sceneHeight);
        }

        if(isRunning){
          for(i5=0;i5<chargeNumber;i5++){
            i=chargePlaces[i5];
            if(AtomElectron[i] != 0 && AtomMoveNow[i]==0){
              forceX = 0.0;
              forceY = 0.0;
                for (i4 = 0; i4 < impurityNumber; i4++){
                  j=impurityPlaces[i4];
                  if(j != i){
                    totalDistance = Math.sqrt(Math.pow(AtomCoords[i][0]-AtomCoords[j][0], 2.0) + Math.pow(AtomCoords[i][1]-AtomCoords[j][1], 2.0));
                    totalForce = 5.0 / Math.pow(totalDistance, 2.0);
                    if((AtomType[j] == 2.0 && AtomElectron[i] == 1.0) |
                      (AtomType[j] == 1.0 && AtomElectron[i] == 2.0)){//Repelling force
                      forceX += totalForce * (AtomCoords[i][0]-AtomCoords[j][0]) / totalDistance;
                      forceY += totalForce * (AtomCoords[j][1]-AtomCoords[i][1]) / totalDistance;
                    }
                    if( (AtomElectron[j] == 2.0 && AtomElectron[i] == 1.0) |
                        (AtomType[j] == AtomElectron[i])){    //Attracting force
                      if(AtomType[j] == AtomElectron[i]){     //Attracting force
                        if (totalDistance < 800.0){           //Electrons/holes are just weakly attracted by nearby impurity atoms
                          totalForce *= 0.005;
                        }
                      }
                      forceX -= totalForce * (AtomCoords[i][0]-AtomCoords[j][0]) / totalDistance;
                      forceY -= totalForce * (AtomCoords[j][1]-AtomCoords[i][1]) / totalDistance;
                    }
                  }
                }
                for (i4 = 0; i4 < chargeNumber; i4++){
                  j=chargePlaces[i4];
                  if(j != i){
                    totalDistance = Math.sqrt(Math.pow(AtomCoords[i][0]-AtomCoords[j][0], 2.0) + Math.pow(AtomCoords[i][1]-AtomCoords[j][1], 2.0));
                    totalForce = 5.0 / Math.pow(totalDistance, 2.0);
                    if(totalDistance < 51.0 && ((AtomElectron[i] == 2 && AtomElectron[j]==1) | (AtomElectron[i] == 1 && AtomElectron[j]==2))
                       && AtomMoveNow[i] == 0 && AtomMoveNow[j] == 0){//Recombination of electron and hole
                      AtomMoveNow[i]=-1 * recombinationTime;
                      AtomMoveNow[j]=-1 * recombinationTime;
                      break;
                    }
                    else{
                      if(AtomElectron[j] == AtomElectron[i]){//Repelling force
                        forceX = forceX + totalForce * (AtomCoords[i][0] - AtomCoords[j][0]) / totalDistance;
                        forceY = forceY + totalForce * (AtomCoords[j][1] - AtomCoords[i][1]) / totalDistance;
                      }
                      if((AtomElectron[j] == 1.0 && AtomElectron[i] == 2.0) |
                        (AtomElectron[j] == 2.0 && AtomElectron[i] == 1.0)){////Attracting force
                        if (totalDistance < 800.0){//Electrons/holes are just weakly attracted by nearby impurity atoms
                          totalForce *= 0.005;
                        }

                        forceX = forceX - totalForce * (AtomCoords[i][0] - AtomCoords[j][0]) / totalDistance;
                        forceY = forceY - totalForce * (AtomCoords[j][1] - AtomCoords[i][1]) / totalDistance;
                      }
                    }
                  }
                }//for (i4 = 0; i4 < chargeNumber; i4++){

//Add forces caused by thermal movement

              forceX += (RandomGenerator.nextInt(100) - 50) * crystalTemperature * 10 / 250000000.0;//random movement caused by temperature
              forceY += (RandomGenerator.nextInt(100) - 50) * crystalTemperature * 10 / 250000000.0;//random movement caused by temperature

              if(AtomMoveNow[i] == 0){//Move electron/hole of the atom
                alpha = Math.atan2(forceY, forceX);
                i2=-1;

                if(directionState[i]==0){
                  if(alpha > 0.523598783333 && alpha < 2.61799391667){
                    i2=i-100;
                  }
                  else if(alpha >= 2.61799391667 | alpha < -1.57079635) {
                    i2=i+99;
                  }
                  else{
                    i2=i+100;
                  }
                }
                if(directionState[i]==1){
                  if(alpha < -0.523598783333 && alpha > -2.61799391667){
                    i2=i+100;
                  }
                  else if(alpha <= -2.61799391667 | alpha > 1.57079635) {
                    i2=i-100;
                  }
                  else{
                    i2=i-99;
                  }
                }
                if(directionState[i]==2){
                  if(alpha > 0.523598783333 && alpha < 2.61799391667){
                    i2=i-100;
                  }
                  else if(alpha >= 2.61799391667 | alpha < -1.57079635) {
                    i2=i+100;
                  }
                  else{
                    i2=i+101;
                  }
                }
                if(directionState[i]==3){
                  if(alpha < -0.523598783333 && alpha > -2.61799391667){
                    i2=i+100;
                  }
                  else if(alpha <= -2.61799391667 | alpha > 1.57079635) {
                    i2=i-101;
                  }
                  else{
                    i2=i-100;
                  }
                }
                if(i2<12200 && i2>-1){//Atom is not at the upper or lower edge of the crystal
                  if(Math.abs(AtomCoords[i2][0] - AtomCoords[i][0]) < 200.0){//Atom is not at the left or right edge of the crystal
                    iNew=-1;
                    for(i4=0;i4<chargeNumber;i4++){
                      if(chargePlacesNew[i4][0]==i2){
                        iNew=-1;
                        break;
                      }
                      if(chargePlaces[i4]==i){
                        iNew=i4;
                      }
                    }
                    if(iNew>-1){
                      if(AtomElectron[i]==1){
                        AtomMoveNow[i2] = RandomGenerator.nextInt(electronMobility);
                      }
                      else{
                        AtomMoveNow[i2] = RandomGenerator.nextInt(holeMobility);
                      }
                      chargePlacesNew[iNew][0]=i2;
                      chargePlacesNew[iNew][1]=AtomElectron[i];
                    }
                  }
                }
              }//if(AtomMoveNow[i] == 0){//No recombination process

            }//if(AtomElectron[i]!=0){
          }//for(i=0;i<12200;i++){

        //inject electrons/holes depending on the external field strength

        if(transistorType=="NPN"){
          if(baseVoltage > 0){
            if(impuritySumZones[5] + chargeSumZones[5] + 10 < baseVoltage / 10 ){
              newElectron = RandomGenerator.nextInt(122) * 100;
              newHole = (RandomGenerator.nextInt(122) + 1) * 100 - (50 - RandomGenerator.nextInt(5));
              i2=0;
              j=0;
              for(i4=0;i4<chargeNumber;i4++){
                if(chargePlacesNew[i4][0] == newElectron | chargePlacesNew[i4][0] == newHole){
                  i2=-1;
                  break;
                }
              }
              if(i2==0){
                for(i4=0;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[0][0];
                    electronInjectY[i4] = AtomCoords[newElectron][1];
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }
                for(i4=j;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[47][0] + 70 + RandomGenerator.nextInt(850);
                    if(RandomGenerator.nextInt(100)>49){
                      electronInjectY[i4] = 200;
                    }
                    else{
                      electronInjectY[i4] = 4950;
                    }
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }

                chargeNumber += 2;
                electronNumber++;
                chargePlacesNew[chargeNumber-1][0] = newElectron;
                chargePlacesNew[chargeNumber-1][1] = 1;
                chargePlacesNew[chargeNumber-1][2] = 0;
                holeNumber++;
                chargePlacesNew[chargeNumber-2][0] = newHole;
                chargePlacesNew[chargeNumber-2][1] = 2;
                chargePlacesNew[chargeNumber-2][2] = 0;
              }
            }//if(impuritySumZones[5] + chargeSumZones[5] + 10 < baseVoltage / 10 ){
          }//if(baseVoltage>0){

          if(collectorVoltage > 0){
            if( impuritySumZones[0] + chargeSumZones[0] > -(collectorVoltage + baseVoltage) / 10 && impuritySumZones[9] + chargeSumZones[9] < collectorVoltage / 10 ){

              newElectron = RandomGenerator.nextInt(122) * 100;
              removeElectron = -1;
              i2=0;
              for(i4=0;i4<chargeNumber;i4++){
                if(chargePlacesNew[i4][0] == newElectron){
                  i2=-1;
                  break;
                }
                if(removeElectron == -1){
                  j = chargePlacesNew[i4][0];
                  while (j > 99) j-=100;
                  if(j > 97){
                    if(chargePlacesNew[i4][1] == 1){
                      removeElectron = i4;
                    }
                  }
                }
              }
              if(i2==0 && removeElectron > -1){
                j=0;
                for(i4=0;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[98][0];
                    electronInjectY[i4] = AtomCoords[chargePlacesNew[removeElectron][0]][1];
                    electronInjectTime[i4] = 25;
                    j=i4;
                    break;
                  }
                }
                for(i4=j;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[0][0];
                    electronInjectY[i4] = AtomCoords[newElectron][1];
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }
                chargePlacesNew[removeElectron][0] = newElectron;
                chargePlacesNew[removeElectron][1] = 1;
                chargePlacesNew[removeElectron][2] = 0;
              }
            }
          }//if(collectorVoltage>0){
        }//if(transistorType=="NPN"){
        else{
          if(baseVoltage > 0){
            if(impuritySumZones[5] + chargeSumZones[5] - 10 > -baseVoltage / 10 ){
              newHole = RandomGenerator.nextInt(122) * 100;
              newElectron = (RandomGenerator.nextInt(122) + 1) * 100 - (50 - RandomGenerator.nextInt(5));
              i2=0;
              j=0;
              for(i4=0;i4<chargeNumber;i4++){
                if(chargePlacesNew[i4][0] == newElectron | chargePlacesNew[i4][0] == newHole){
                  i2=-1;
                  break;
                }
              }
              if(i2==0){
                for(i4=0;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[0][0];
                    electronInjectY[i4] = AtomCoords[newHole][1];
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }
                for(i4=j;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[50][0] + 50 + RandomGenerator.nextInt(750);
                    if(RandomGenerator.nextInt(100)>49){
                      electronInjectY[i4] = 200;
                    }
                    else{
                      electronInjectY[i4] = 4950;
                    }
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }

                chargeNumber += 2;
                electronNumber++;
                chargePlacesNew[chargeNumber-1][0] = newElectron;
                chargePlacesNew[chargeNumber-1][1] = 1;
                chargePlacesNew[chargeNumber-1][2] = 0;
                holeNumber++;
                chargePlacesNew[chargeNumber-2][0] = newHole;
                chargePlacesNew[chargeNumber-2][1] = 2;
                chargePlacesNew[chargeNumber-2][2] = 0;
              }
            }//if(impuritySumZones[5] + chargeSumZones[5] - 10 < baseVoltage / 10 ){
          }//if(baseVoltage>0){

          if(collectorVoltage > 0){
            if( impuritySumZones[0] + chargeSumZones[0] < (collectorVoltage + baseVoltage) / 10 && impuritySumZones[9] + chargeSumZones[9] > -collectorVoltage / 10 ){

              newHole = RandomGenerator.nextInt(122) * 100;
              removeElectron = -1;
              i2=0;
              for(i4=0;i4<chargeNumber;i4++){
                if(chargePlacesNew[i4][0] == newHole){
                  i2=-1;
                  break;
                }
                if(removeElectron == -1){
                  j = chargePlacesNew[i4][0];
                  while (j > 99) j-=100;
                  if(j > 97){
                    if(chargePlacesNew[i4][1] == 2){
                      removeElectron = i4;
                    }
                  }
                }
              }
              if(i2==0 && removeElectron > -1){
                j=0;
                for(i4=0;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[98][0];
                    electronInjectY[i4] = AtomCoords[chargePlacesNew[removeElectron][0]][1];
                    electronInjectTime[i4] = 25;
                    j=i4;
                    break;
                  }
                }
                for(i4=j;i4<100;i4++){
                  if(electronInjectTime[i4]==0){
                    electronInjectX[i4] = AtomCoords[0][0];
                    electronInjectY[i4] = AtomCoords[newHole][1];
                    electronInjectTime[i4] = 25;
                    break;
                  }
                }
                chargePlacesNew[removeElectron][0] = newHole;
                chargePlacesNew[removeElectron][1] = 2;
                chargePlacesNew[removeElectron][2] = 0;
              }
            }
          }//if(collectorVoltage>0){
        }





          for(i4=0;i4<chargeNumber;i4++){
            AtomElectron[chargePlaces[i4]]=0;
          }
          for(i4=0;i4<chargeNumber;i4++){//Remove electrons/holes after recombination process
            if(AtomMoveNow[chargePlacesNew[i4][0]]==-1){
              AtomMoveNow[chargePlacesNew[i4][0]]=0;
              if(chargePlacesNew[i4][1]==1){
                electronNumber--;
              }
              else{
                holeNumber--;
              }
              chargePlacesNew[i4][0]=chargePlacesNew[chargeNumber-1][0];
              chargePlacesNew[i4][1]=chargePlacesNew[chargeNumber-1][1];
              chargePlacesNew[i4][2]=chargePlacesNew[chargeNumber-1][2];
              chargeNumber--;
              i4--;
            }
          }
          for(i=0;i<10;i++){
            chargeSumZones[i]=0;
          }

          //Update chargePlaces
          for(i4=0;i4<chargeNumber;i4++){
            chargePlaces[i4]=chargePlacesNew[i4][0];

            j=chargePlacesNew[i4][0];
            while(j>99)j-=100;

            if(chargePlacesNew[i4][1]==1){
              if(j < 10){
                chargeSumZones[0]--;
              }
              if(j >= 10 && j < 20){
                chargeSumZones[1]--;
              }
              if(j >= 20 && j < 30){
                chargeSumZones[2]--;
              }
              if(j >= 30 && j < 40){
                chargeSumZones[3]--;
              }
              if(j >= 40 && j < 50){
                chargeSumZones[4]--;
              }
              if(j >= 50 && j < 60){
                chargeSumZones[5]--;
              }
              if(j >= 60 && j < 70){
                chargeSumZones[6]--;
              }
              if(j >= 70 && j < 80){
                chargeSumZones[7]--;
              }
              if(j >= 80 && j < 90){
                chargeSumZones[8]--;
              }
              if(j >= 90){
                chargeSumZones[9]--;
              }
            }
            if(chargePlacesNew[i4][1]==2){
              if(j < 10){
                chargeSumZones[0]++;
              }
              if(j >= 10 && j < 20){
                chargeSumZones[1]++;
              }
              if(j >= 20 && j < 30){
                chargeSumZones[2]++;
              }
              if(j >= 30 && j < 40){
                chargeSumZones[3]++;
              }
              if(j >= 40 && j < 50){
                chargeSumZones[4]++;
              }
              if(j >= 50 && j < 60){
                chargeSumZones[5]++;
              }
              if(j >= 60 && j < 70){
                chargeSumZones[6]++;
              }
              if(j >= 70 && j < 80){
                chargeSumZones[7]++;
              }
              if(j >= 80 && j < 90){
                chargeSumZones[8]++;
              }
              if(j >= 90){
                chargeSumZones[9]++;
              }
            }

            AtomElectron[chargePlacesNew[i4][0]]=chargePlacesNew[i4][1];
            if(chargePlacesNew[i4][2]>0){
              chargePlacesNew[i4][2]--;
            }
          }
        }//if(isRunning)


        bufferImageSurface.setColor(new Color(0,0,0));
        bufferImageSurface.drawRect(0, 0, sceneWidth-1, sceneHeight-1);

        //paint bufferImage and make it visible
        g2.drawImage(bufferImage, 0, 0, this);
        g2.setComposite(origComposite);


      if(savePics == 1){
//        picNumber++;
        picName="picsequence/drawing-" + new DecimalFormat("000000").format(picNumber) + ".png";
        try{
          ImageIO.write(bufferImage, "PNG", new File(picName));
        }
        catch (IOException e1) {
          e1.printStackTrace();
        }
      }

      crystalSize2=crystalSize;

    }
}

